<?php
require_once '../config/config.php';

if (!isAdmin() && getUserRole() !== 'pemdes') {
    redirect('../dashboard.php');
}

$db = Database::getInstance()->getConnection();

// Handle optimize
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'optimize') {
    $tanggal = sanitize($_POST['tanggal'] ?? date('Y-m-d'));
    $depo_id = intval($_POST['depo_id'] ?? 0);
    
    // Get pending distributions with prediction data
    $stmt = $db->prepare("SELECT dt.*, dus.latitude as dusun_lat, dus.longitude as dusun_lng, 
                         dus.nama_dusun, des.nama_desa,
                         p.tingkat_risiko, p.skor_risiko,
                         (SELECT COUNT(*) FROM laporan_warga lw WHERE lw.dusun_id = dus.id AND lw.status = 'verified' AND lw.tanggal >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)) as jumlah_laporan
                         FROM distribusi_tangki dt
                         JOIN dusun dus ON dt.dusun_id = dus.id
                         JOIN desa des ON dus.desa_id = des.id
                         LEFT JOIN prediksi_kekeringan p ON dus.id = p.dusun_id AND p.tanggal_prediksi >= CURDATE()
                         WHERE dt.status = 'disetujui' AND dt.tanggal_jadwal = ? AND dus.latitude IS NOT NULL AND dus.longitude IS NOT NULL");
    $stmt->execute([$tanggal]);
    $distributions = $stmt->fetchAll();
    
    if (empty($distributions)) {
        $_SESSION['error'] = 'Tidak ada distribusi yang perlu dioptimasi';
        redirect('optimasi.php');
    }
    
    // Get depo
    $stmt = $db->prepare("SELECT * FROM depo_pengiriman WHERE id = ? AND is_active = 1");
    $stmt->execute([$depo_id]);
    $depo = $stmt->fetch();
    
    if (!$depo) {
        $_SESSION['error'] = 'Depo tidak ditemukan';
        redirect('optimasi.php');
    }
    
    // Calculate distance and optimize route
    $depo_lat = floatval($depo['latitude']);
    $depo_lng = floatval($depo['longitude']);
    
    // Calculate priority score for each distribution
    // Priority = (Condition Score * 0.5) + (Distance Score * 0.3) + (Report Score * 0.2)
    foreach ($distributions as &$dist) {
        $dusun_lat = floatval($dist['dusun_lat']);
        $dusun_lng = floatval($dist['dusun_lng']);
        
        // Calculate distance from depo
        $earth_radius = 6371; // km
        $lat_diff = deg2rad($dusun_lat - $depo_lat);
        $lng_diff = deg2rad($dusun_lng - $depo_lng);
        $a = sin($lat_diff/2) * sin($lat_diff/2) +
             cos(deg2rad($depo_lat)) * cos(deg2rad($dusun_lat)) *
             sin($lng_diff/2) * sin($lng_diff/2);
        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        $distance = $earth_radius * $c;
        
        $dist['jarak_km'] = round($distance, 2);
        
        // Condition Score (based on prediction)
        $condition_score = 0;
        if ($dist['tingkat_risiko'] === 'kering') {
            $condition_score = 100; // Highest priority
        } elseif ($dist['tingkat_risiko'] === 'waspada') {
            $condition_score = 60;
        } elseif ($dist['tingkat_risiko'] === 'aman') {
            $condition_score = 20;
        } else {
            $condition_score = 30; // No prediction, medium priority
        }
        
        // Distance Score (closer = higher score, max 50km = 0 score)
        $max_distance = 50; // km
        $distance_score = max(0, 100 - ($distance / $max_distance * 100));
        
        // Report Score (more reports = higher priority)
        $report_count = intval($dist['jumlah_laporan'] ?? 0);
        $report_score = min(100, $report_count * 20); // Max 5 reports = 100
        
        // Calculate priority score (higher = more urgent)
        $priority_score = ($condition_score * 0.5) + ($distance_score * 0.3) + ($report_score * 0.2);
        
        $dist['condition_score'] = $condition_score;
        $dist['distance_score'] = $distance_score;
        $dist['report_score'] = $report_score;
        $dist['priority_score'] = round($priority_score, 2);
    }
    
    // Sort by priority score (highest first) - prioritize urgent conditions and nearby locations
    usort($distributions, function($a, $b) {
        // First sort by priority score (descending)
        if ($a['priority_score'] != $b['priority_score']) {
            return $b['priority_score'] <=> $a['priority_score'];
        }
        // If same priority, sort by distance (nearest first)
        return $a['jarak_km'] <=> $b['jarak_km'];
    });
    
    // Optimize route using Nearest Neighbor Algorithm after priority sorting
    // This ensures we visit locations in priority order but minimize travel distance
    $optimized_route = [];
    $visited = [];
    $current_lat = $depo_lat;
    $current_lng = $depo_lng;
    $urutan = 1;
    
    // Start with highest priority items
    while (count($optimized_route) < count($distributions)) {
        $nearest = null;
        $nearest_distance = null;
        $nearest_index = null;
        
        // Find nearest unvisited location from current position
        foreach ($distributions as $index => $dist) {
            if (isset($visited[$index])) continue;
            
            $dusun_lat = floatval($dist['dusun_lat']);
            $dusun_lng = floatval($dist['dusun_lng']);
            
            $lat_diff = deg2rad($dusun_lat - $current_lat);
            $lng_diff = deg2rad($dusun_lng - $current_lng);
            $a = sin($lat_diff/2) * sin($lat_diff/2) +
                 cos(deg2rad($current_lat)) * cos(deg2rad($dusun_lat)) *
                 sin($lng_diff/2) * sin($lng_diff/2);
            $c = 2 * atan2(sqrt($a), sqrt(1-$a));
            $distance = $earth_radius * $c;
            
            // For first item, prefer highest priority
            // For subsequent items, prefer nearest but still consider priority
            if ($nearest === null || 
                (count($optimized_route) === 0 && $dist['priority_score'] > ($nearest['priority_score'] ?? 0)) ||
                (count($optimized_route) > 0 && $distance < $nearest_distance)) {
                $nearest = $dist;
                $nearest_distance = $distance;
                $nearest_index = $index;
            }
        }
        
        if ($nearest === null) break;
        
        $visited[$nearest_index] = true;
        $optimized_route[] = $nearest;
        
        // Calculate segment distance and BBM
        $segment_distance = $nearest_distance;
        $bbm_per_km = 0.2; // liter per km
        $estimasi_bbm = $segment_distance * $bbm_per_km;
        $biaya_bbm = $estimasi_bbm * 10000; // Assuming 10,000 per liter
        
        // Update distribution with optimized route order
        $stmt = $db->prepare("UPDATE distribusi_tangki SET 
                             depo_id = ?, jarak_km = ?, estimasi_bbm_liter = ?, 
                             biaya_bbm = ?, urutan_rute = ? 
                             WHERE id = ?");
        $stmt->execute([
            $depo_id,
            round($segment_distance, 2),
            round($estimasi_bbm, 2),
            round($biaya_bbm, 2),
            $urutan,
            $nearest['id']
        ]);
        
        $current_lat = floatval($nearest['dusun_lat']);
        $current_lng = floatval($nearest['dusun_lng']);
        $urutan++;
    }
    
    // Calculate total distance
    $total_distance = 0;
    $previous_lat = $depo_lat;
    $previous_lng = $depo_lng;
    
    foreach ($optimized_route as $dist) {
        $lat_diff = deg2rad(floatval($dist['dusun_lat']) - $previous_lat);
        $lng_diff = deg2rad(floatval($dist['dusun_lng']) - $previous_lng);
        $a = sin($lat_diff/2) * sin($lat_diff/2) +
             cos(deg2rad($previous_lat)) * cos(deg2rad(floatval($dist['dusun_lat']))) *
             sin($lng_diff/2) * sin($lng_diff/2);
        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        $segment_distance = $earth_radius * $c;
        $total_distance += $segment_distance;
        
        $previous_lat = floatval($dist['dusun_lat']);
        $previous_lng = floatval($dist['dusun_lng']);
    }
    
    // Add return to depo distance
    $lat_diff = deg2rad($depo_lat - $previous_lat);
    $lng_diff = deg2rad($depo_lng - $previous_lng);
    $a = sin($lat_diff/2) * sin($lat_diff/2) +
         cos(deg2rad($previous_lat)) * cos(deg2rad($depo_lat)) *
         sin($lng_diff/2) * sin($lng_diff/2);
    $c = 2 * atan2(sqrt($a), sqrt(1-$a));
    $return_distance = $earth_radius * $c;
    $total_distance += $return_distance;
    
    $_SESSION['success'] = "Optimasi berhasil! Total jarak: " . round($total_distance, 2) . " km. Rute telah dioptimasi berdasarkan prioritas kondisi warga, jarak, dan efisiensi BBM.";
    redirect('optimasi.php');
}

// Get depo list
$stmt = $db->query("SELECT * FROM depo_pengiriman WHERE is_active = 1 ORDER BY nama_depo");
$depo_list = $stmt->fetchAll();

// Get distributions ready for optimization
$stmt = $db->query("SELECT dt.*, dus.nama_dusun, des.nama_desa, d.nama_depo
                    FROM distribusi_tangki dt
                    JOIN dusun dus ON dt.dusun_id = dus.id
                    JOIN desa des ON dus.desa_id = des.id
                    LEFT JOIN depo_pengiriman d ON dt.depo_id = d.id
                    WHERE dt.status = 'disetujui'
                    ORDER BY dt.tanggal_jadwal, dt.urutan_rute");
$distributions = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AI Optimasi Pengiriman - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <?php include '../includes/sidebar.php'; ?>
    
    <main class="main-content">
        <header class="content-header">
            <div class="header-left">
                <h1><i class="fas fa-route"></i> AI Optimasi Pengiriman</h1>
                <p>Optimasi rute pengiriman berdasarkan jarak dan hemat BBM</p>
            </div>
        </header>

        <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i> <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
        <?php endif; ?>

        <!-- Optimize Form -->
        <div class="content-section">
            <h2><i class="fas fa-brain"></i> Optimasi Rute</h2>
            <form method="POST" class="form-container">
                <input type="hidden" name="action" value="optimize">
                <div class="form-row">
                    <div class="form-group">
                        <label>Tanggal Pengiriman</label>
                        <input type="date" name="tanggal" value="<?php echo date('Y-m-d'); ?>" required>
                    </div>
                    <div class="form-group">
                        <label>Depo Pengiriman</label>
                        <select name="depo_id" required>
                            <option value="">Pilih Depo</option>
                            <?php foreach ($depo_list as $depo): ?>
                            <option value="<?php echo $depo['id']; ?>"><?php echo htmlspecialchars($depo['nama_depo']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-magic"></i> Optimasi Rute
                </button>
            </form>
            <div class="alert alert-info" style="margin-top: 20px;">
                <i class="fas fa-info-circle"></i> 
                <strong>Cara Kerja AI Optimasi Cerdas:</strong><br>
                1. <strong>Prioritas Kondisi Warga (50%):</strong> Daerah dengan status "Kering" mendapat prioritas tertinggi<br>
                2. <strong>Jarak dari Depo (30%):</strong> Daerah terdekat dari depo diprioritaskan untuk efisiensi<br>
                3. <strong>Laporan Warga (20%):</strong> Daerah dengan laporan lebih banyak mendapat prioritas<br>
                4. <strong>Optimasi Rute:</strong> Setelah prioritas, rute dioptimasi untuk meminimalkan jarak tempuh<br>
                5. <strong>Hemat BBM:</strong> Menghitung estimasi BBM (0.2 L/km) dan biaya pengiriman
            </div>
        </div>

        <!-- Optimized Routes -->
        <div class="content-section">
            <h2>Rute yang Telah Dioptimasi</h2>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Urutan</th>
                            <th>Tanggal</th>
                            <th>Desa/Dusun</th>
                            <th>Kondisi</th>
                            <th>Prioritas</th>
                            <th>Depo</th>
                            <th>Jarak (km)</th>
                            <th>Estimasi BBM (L)</th>
                            <th>Biaya BBM</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($distributions)): ?>
                        <tr>
                            <td colspan="10" class="text-center">Belum ada rute yang dioptimasi</td>
                        </tr>
                        <?php else: ?>
                        <?php 
                        $total_jarak = 0;
                        $total_bbm = 0;
                        $total_biaya = 0;
                        foreach ($distributions as $dist): 
                            $total_jarak += floatval($dist['jarak_km'] ?? 0);
                            $total_bbm += floatval($dist['estimasi_bbm_liter'] ?? 0);
                            $total_biaya += floatval($dist['biaya_bbm'] ?? 0);
                        ?>
                        <tr>
                            <td>
                                <?php if ($dist['urutan_rute'] > 0): ?>
                                <span class="badge badge-primary"><?php echo $dist['urutan_rute']; ?></span>
                                <?php else: ?>
                                <span class="badge badge-warning">-</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo formatDate($dist['tanggal_jadwal']); ?></td>
                            <td><?php echo htmlspecialchars($dist['nama_desa']); ?> - <?php echo htmlspecialchars($dist['nama_dusun']); ?></td>
                            <td>
                                <?php if ($dist['tingkat_risiko']): ?>
                                <span class="badge badge-<?php echo $dist['tingkat_risiko'] === 'kering' ? 'danger' : ($dist['tingkat_risiko'] === 'waspada' ? 'warning' : 'success'); ?>">
                                    <?php echo ucfirst($dist['tingkat_risiko']); ?>
                                </span>
                                <?php else: ?>
                                <span class="badge badge-info">Belum ada prediksi</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="badge badge-primary" title="Skor Prioritas: <?php echo number_format($dist['priority_score'] ?? 0, 2); ?>">
                                    <?php echo number_format($dist['priority_score'] ?? 0, 1); ?>
                                </span>
                                <?php if (isset($dist['jumlah_laporan']) && $dist['jumlah_laporan'] > 0): ?>
                                <br><small style="color: #666;"><?php echo $dist['jumlah_laporan']; ?> laporan</small>
                                <?php endif; ?>
                            </td>
                            <td><?php echo htmlspecialchars($dist['nama_depo'] ?? '-'); ?></td>
                            <td><?php echo number_format($dist['jarak_km'] ?? 0, 2); ?></td>
                            <td><?php echo number_format($dist['estimasi_bbm_liter'] ?? 0, 2); ?></td>
                            <td>Rp <?php echo number_format($dist['biaya_bbm'] ?? 0, 0, ',', '.'); ?></td>
                            <td>
                                <span class="badge badge-success">Disetujui</span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <tr style="background: #f8f9fa; font-weight: bold;">
                            <td colspan="6">TOTAL</td>
                            <td><?php echo number_format($total_jarak, 2); ?> km</td>
                            <td><?php echo number_format($total_bbm, 2); ?> L</td>
                            <td>Rp <?php echo number_format($total_biaya, 0, ',', '.'); ?></td>
                            <td></td>
                        </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <script src="../assets/js/main.js"></script>
</body>
</html>

